/**
 * Service for 3DS Secure authentication
 */
class ThreeDSService {
  constructor() {
    this.authenticationRequestBody = {};
    this.callback = undefined;
    this.host = undefined;
    this.authToken = undefined;
    this.requestId = undefined;
    this.formChallenge = undefined;
    this.iframeChallenge = undefined;
    this.timesToRetry = 0;
    this.intervalHandler = undefined;
    this.challengeReturnData = null;
    this.proxyBaseUrl = undefined;
    this.setupMessageListener();
  }

  /**
   * Sets up the listener for 3DS iframe messages
   */
  setupMessageListener() {
    window.addEventListener('message', async (event) => {
      console.log('3DS event received:', event);
      if (event.origin === this.host) {
        console.log("Threeds event received");
        let data = JSON.parse(event.data);
        console.log('Merchant received a message:', data);
        if (data !== undefined && data.Status) {
          await this.doAuthentication(this.authenticationRequestBody);
        }
      } else {
        console.log('Message from different origin.');
      }
    });
  }

  /**
   * Gets device information
   * @returns {Object} - Device information
   */
  getDeviceInformation() {
    return {
      ipAddress: navigator.ipAddress,
      httpAcceptBrowserValue: navigator.acceptHeader,
      httpAcceptContent: navigator.httpAcceptContent,
      httpBrowserLanguage: navigator.language,
      httpBrowserJavaEnabled: navigator.javaEnabled(),
      httpBrowserJavaScriptEnabled: navigator.httpBrowserJavaScriptEnabled,
      httpBrowserColorDepth: screen.colorDepth,
      httpBrowserScreenHeight: screen.height,
      httpBrowserScreenWidth: screen.width,
      httpBrowserTimeDifference: new Date().getTimezoneOffset(),
      userAgentBrowserValue: navigator.userAgent,
    };
  }

  /**
   * Collects device data
   * @param {Object} setupResponse - Authentication setup response
   */
  collectData(setupResponse) {
    this.requestId = setupResponse.requestId;

    let iframeMethod = document.createElement('iframe');
    iframeMethod.id = 'iframe_method';
    iframeMethod.name = 'iframe_method';
    iframeMethod.style.display = 'none';
    iframeMethod.style.position = 'absolute';
    iframeMethod.style.left = '-9999px';
    iframeMethod.style.top = '-9999px';
    iframeMethod.style.width = '1px';
    iframeMethod.style.height = '1px';
    document.body.appendChild(iframeMethod);

    let form = document.createElement('form');
    form.setAttribute('method', 'POST');
    form.setAttribute('action', setupResponse.deviceDataCollectionUrl);
    form.setAttribute('target', 'iframe_method');
    
    let threeDSMethodData = document.createElement('input');
    threeDSMethodData.setAttribute('type', 'hidden');
    threeDSMethodData.setAttribute('name', 'JWT');
    threeDSMethodData.setAttribute('value', setupResponse.accessToken);
    form.appendChild(threeDSMethodData);
    form.style.display = 'none';
    document.body.appendChild(form);

    form.submit();
  }

  /**
   * Handles error responses
   * @param {string} url - Request URL
   * @param {Object|string} responseBody - Error response body
   */
  handleErrorResponse(url, responseBody) {
    console.log("Error response from " + url);
    if (responseBody !== undefined) {  
      console.log(responseBody);
    }
    
    // Disable loading state on error
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
      submitBtn.classList.remove('btn-loading');
      submitBtn.disabled = false;
      submitBtn.textContent = 'Start Challenge';
    }
    
    this.callback();
  }

  /**
   * Assembles authentication request
   * @param {Object} cardInfo - Card information
   * @param {Object} generalInformation - General information
   * @param {Object} orderInformation - Order information
   * @param {Object} buyerInformation - Buyer information
   * @param {string} merchantUrl - Merchant URL
   * @param {string} returnUrl - Return URL
   * @returns {Object} - Request body
   */
  assembleAuthenticateRequest(cardInfo, generalInformation, orderInformation, buyerInformation, merchantUrl, returnUrl) {
    const deviceInformation = this.getDeviceInformation();

    return {
      ...cardInfo,
      ...generalInformation,
      orderInformation,
      buyerInformation,
      merchantUrl,
      deviceInformation,
      returnUrl,
    };
  }


 assembleThreeDsData(response) {
  return {
    xid: response.xid,
    cavv: response.cavv,
    secure_version: response.specificationVersion,
    directory_server_transaction_id: response.directoryServerTransactionId,
    three_ds_server_transaction_id: response.threeDSServerTransactionId,
  };
}

    /**
   * Executes authentication setup
   * @param {Object} requestBodySetup - Setup data
   * @returns {Promise<void>}
   */
    async doAuthenticationSetup(requestBodySetup) {
      const url = `${this.proxyBaseUrl}/api/proxy/v2/threeds-authentication-setup`;
      
      console.log('doAuthenticationSetup:', url);
      try {
        const response = await fetch(url, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json; charset=utf-8',
            'Authorization': this.authToken,
          },
          body: JSON.stringify(requestBodySetup)
        });
  
        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
  
        const result = await response.text();
        const responseData = JSON.parse(result);
        console.log('doAuthenticationSetup responseData:', responseData);
        if (responseData.code === "WAITING_3DS_AUTHENTICATION") {
          this.authenticationRequestBody["requestId"] = responseData.requestId;
          this.authenticationRequestBody["referenceId"] = responseData.referenceId;
          this.host = responseData.deviceDataCollectionUrl.split('.com')[0] + '.com';
          this.collectData(responseData);
        } else {
          this.handleErrorResponse(url, responseData);
        }
      } catch (error) {
        console.error('Error in authentication setup:', error);
        this.handleErrorResponse(url, error.message);
      }
    }

  /**
   * Executes authentication
   * @param {Object} bodyAuthentication - Authentication data
   * @returns {Promise<void>}
   */
  async doAuthentication(bodyAuthentication) {
    const url = `${this.proxyBaseUrl}/api/proxy/v2/threeds-authentication`;
    
    console.log('doAuthentication:', url);
    try {
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json; charset=utf-8',
          'Authorization': this.authToken,
        },
        body: JSON.stringify(bodyAuthentication)
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.text();
      const responseData = JSON.parse(result);
      console.log('doAuthentication responseData:', responseData);
      if (responseData.code === "ACCEPTED") {
        const threeDsData = this.assembleThreeDsData(responseData);

        console.log('threeDsData:', threeDsData);
        
         // Clear 3DS return data from server after successful processing
         await this.clearThreeDsReturnData();

        // Displays 3DS data in HTML
        this.displayThreeDsResults(threeDsData);
        
      } else if (responseData.code === "WAITING_3DS_AUTHENTICATION") {
        this.doChallenge(responseData);
      } else {
        this.handleErrorResponse(url, responseData);
      }
    } catch (error) {
      console.error('Error in authentication:', error);
      this.handleErrorResponse(url, error.message);
    }
  }

  /**
   * Executes 3DS challenge
   * @param {Object} authenticationResponse - Authentication response
   */
  doChallenge(authenticationResponse) {  
    // Shows the 3DS container
    const container = document.getElementById('3ds-container');
    const iframeWrapper = document.getElementById('iframe-wrapper');
    
    if (container) {
      container.style.display = 'block';
    }

    // Remove loading from button when challenge starts
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
      submitBtn.classList.remove('btn-loading');
      submitBtn.textContent = 'Start Challenge';
      submitBtn.disabled = false;
    }

    this.iframeChallenge = document.createElement('iframe');
    this.iframeChallenge.id = 'step-up-iframe';
    this.iframeChallenge.name = 'step-up-iframe';

    // Adds iframe to wrapper
    if (iframeWrapper) {
      iframeWrapper.appendChild(this.iframeChallenge);
    } else {
      // Fallback: if wrapper does not exist, adds to body
      document.body.appendChild(this.iframeChallenge);
    }

    this.formChallenge = document.createElement('form');
    this.formChallenge.setAttribute('id', 'step-up-form');
    this.formChallenge.setAttribute('method', 'POST');
    this.formChallenge.setAttribute('action', authenticationResponse.stepUpUrl);
    this.formChallenge.setAttribute('target', 'step-up-iframe');
    this.formChallenge.style.display = 'none';

    let jwtData = document.createElement('input');
    jwtData.setAttribute('type', 'hidden');
    jwtData.setAttribute('name', 'JWT');
    jwtData.setAttribute('value', authenticationResponse.accessToken);
    this.formChallenge.appendChild(jwtData);
    document.body.append(this.formChallenge);
    
    this.formChallenge.submit();
    this.timesToRetry = 0;
    this.intervalHandler = setInterval(() => this.waitChallengeResult(), 2000);
  }

  /**
   * Waits for challenge result
   * @returns {Promise<void>}
   */
  async waitChallengeResult() {  
    try {     
        console.log("Calling returnUrl:", this.authenticationRequestBody.returnUrl);


      const response = await fetch(this.authenticationRequestBody.returnUrl, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json; charset=utf-8',
        }
      });

      const result = await response.json();

      if (!result.success || !result.data || !result.data.TransactionId) {
        this.handleWait();
        return;
      }

     
      clearInterval(this.intervalHandler);
      this.hideChallenge();
      await this.getChallengeResults({
        authenticationTransactionId: result.data.TransactionId,
        requestId: this.requestId
      });
      
     
    } catch (error) {
      console.error('Error waiting for challenge result:', error);
      this.handleWait();
    }
  }

  /**
   * Hides the challenge
   */
  hideChallenge() {
    if (this.iframeChallenge) {
      this.iframeChallenge.style.display = 'none';
    }
    if (this.formChallenge) {
      this.formChallenge.style.display = 'none';
    }
    
    // Removes iframe from wrapper
    const iframeWrapper = document.getElementById('iframe-wrapper');
    if (iframeWrapper && this.iframeChallenge && iframeWrapper.contains(this.iframeChallenge)) {
      iframeWrapper.removeChild(this.iframeChallenge);
    } else if (this.iframeChallenge && document.body.contains(this.iframeChallenge)) {
      document.body.removeChild(this.iframeChallenge);
    }
    
    if (this.formChallenge && document.body.contains(this.formChallenge)) {
      document.body.removeChild(this.formChallenge);
    }
    
    // Hides the 3DS container
    const container = document.getElementById('3ds-container');
    if (container) {
      container.style.display = 'none';
    }
  }

  /**
   * Handles wait time
   */
  handleWait() {
    if (this.timesToRetry > 30) {
      clearInterval(this.intervalHandler);
      this.hideChallenge();
      console.log("Timed out");
      this.handleErrorResponse(this.authenticationRequestBody.returnUrl, "No challenge received");
    } else {
      this.timesToRetry = this.timesToRetry + 1;
    }
  }

  /**
   * Gets challenge results
   * @param {Object} challengeBody - Challenge data
   * @returns {Promise<void>}
   */
  async getChallengeResults(challengeBody) {
    const url = `${this.proxyBaseUrl}/api/proxy/v2/threeds-challenge-result`;
    
    console.log('getChallengeResults:', url);
    try {
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json; charset=utf-8',
          'Authorization': this.authToken,
        },
        body: JSON.stringify(challengeBody)
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.text();
      const responseData = JSON.parse(result);
      console.log('getChallengeResults responseData:', responseData);
      if (responseData.code === "ACCEPTED") {
        const threeDsData = this.assembleThreeDsData(responseData);

        console.log('threeDsData:', threeDsData);
        
        // Clear 3DS return data from server after successful processing
        await this.clearThreeDsReturnData();

        // Displays 3DS data in HTML
        this.displayThreeDsResults(threeDsData);
      } else {
        this.handleErrorResponse(url, responseData);
      }
    } catch (error) {
      console.error('Error getting challenge results:', error);
      this.handleErrorResponse(url, error.message);
    }
  }

  /**
   * Clears 3DS return data from server
   * @returns {Promise<void>}
   */
  async clearThreeDsReturnData() {
    try {
      const url = `${this.proxyBaseUrl}/api/3ds-return`;
      console.log('Clearing 3DS return data:', url);
      
      const response = await fetch(url, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json; charset=utf-8',
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.json();
      console.log('3DS return data cleared:', result);
    } catch (error) {
      console.error('Error clearing 3DS return data:', error);
    }
  }

  /**
   * Displays 3DS results in HTML
   * @param {Object} threeDsData - 3DS data
   */
  displayThreeDsResults(threeDsData) {
    // Fills data in existing HTML elements
    document.getElementById('json-result').textContent = JSON.stringify(threeDsData, null, 2);
    
    // Activates results section
    const resultsContainer = document.getElementById('results-container');
    if (resultsContainer) {
      resultsContainer.style.display = 'block';
    }
    
    // Blocks the start challenge button and removes loading state
    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
      submitBtn.classList.remove('btn-loading');
      submitBtn.disabled = true;
      submitBtn.textContent = 'Process Completed';
      submitBtn.style.background = '#6c757d';
      submitBtn.style.cursor = 'not-allowed';
    }
  }

  /**
   * Starts 3DS process
   * @param {Object} cardInfo - Card information
   * @param {Object} billTo - Billing information
   * @param {Object} amountDetails - Amount details
   * @param {string} mobilePhone - Mobile phone
   * @param {string} merchantUrl - Merchant URL
   * @param {string} returnUrl - Return URL
   * @param {Object} generalInformation - General information
   * @param {Function} callbackFunction - Callback function
   * @param {string} authorizationToken - Authorization token
   * @param {string} proxyBaseUrl - Proxy base URL
   * @returns {boolean} - Operation success
   */
  async doThreeDs(cardInfo, billTo, amountDetails, mobilePhone, merchantUrl, returnUrl, generalInformation, callbackFunction, authorizationToken, proxyBaseUrl) {
    if (cardInfo === undefined || 
        billTo === undefined || 
        amountDetails === undefined || 
        mobilePhone === undefined || 
        merchantUrl === undefined || 
        returnUrl === undefined || 
        generalInformation === undefined || 
        callbackFunction === undefined || 
        authorizationToken === undefined || 
        proxyBaseUrl === undefined) {
      return false;
    }
    
    this.authToken = authorizationToken;
    this.proxyBaseUrl = proxyBaseUrl;
    this.authenticationRequestBody = this.assembleAuthenticateRequest(
      cardInfo, 
      generalInformation, 
      { billTo, amountDetails }, 
      { mobilePhone }, 
      merchantUrl, 
      returnUrl
    );
    this.callback = callbackFunction;
    
    await this.doAuthenticationSetup(cardInfo);
    return true;
  }
}

// Global service instance
const threeDSService = new ThreeDSService();

// Function to start 3DS process
async function doThreeDs(cardInfo, billTo, amountDetails, mobilePhone, merchantUrl, returnUrl, generalInformation, callbackFunction, authorizationToken, proxyBaseUrl) {
  return await threeDSService.doThreeDs(
    cardInfo, 
    billTo, 
    amountDetails, 
    mobilePhone, 
    merchantUrl, 
    returnUrl, 
    generalInformation, 
    callbackFunction, 
    authorizationToken, 
    proxyBaseUrl
  );
}
