/**
 * Service to fetch transactionalToken from ZroBank API
 */
class ZroBankTokenService {
  constructor() {
    this.accessToken = null;
  }

  /**
   * Generates a UUID v4
   * @returns {string} - Generated UUID
   */
  generateUUID() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
      const r = Math.random() * 16 | 0;
      const v = c == 'x' ? r : (r & 0x3 | 0x8);
      return v.toString(16);
    });
  }


  /**
   * Fetches transactionalToken from ZroBank API
   * @param {string} zroBankBaseUrl - ZroBank base URL
   * @param {string} zroBankApiId - ZroBank API ID
   * @param {string} zroBankApiKey - ZroBank API Key
   * @returns {Promise<string>} - The transactionalToken
   */
  async getTransactionalTokenZroBank(zroBankBaseUrl, zroBankApiId, zroBankApiKey) {
    try {
      // First, get access token from signin
      const accessToken = await this.getAccessToken(zroBankBaseUrl, zroBankApiId, zroBankApiKey);
      
      // Then, get transactional token
      const transactionalToken = await this.getTransactionalToken(zroBankBaseUrl, accessToken);
      
      return transactionalToken;
    } catch (error) {
      console.error('Error fetching ZroBank transactionalToken:', error);
      throw new Error('Failed to get ZroBank transactionalToken: ' + error.message);
    }
  }

  /**
   * Gets access token from ZroBank signin endpoint
   * @param {string} zroBankBaseUrl - ZroBank base URL
   * @param {string} zroBankApiId - ZroBank API ID
   * @param {string} zroBankApiKey - ZroBank API Key
   * @returns {Promise<string>} - The access token
   */
  async getAccessToken(zroBankBaseUrl, zroBankApiId, zroBankApiKey) {
    const nonce = this.generateUUID();
    const signinUrl = `${zroBankBaseUrl}/v1/auth/signin`;
    
    console.log('Getting access token from ZroBank signin...', signinUrl);
    
    const response = await fetch(signinUrl, {
      method: 'POST',
      headers: {
        'accept': 'application/json',
        'nonce': nonce,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        api_id: zroBankApiId,
        api_key: zroBankApiKey
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const { data } = await response.json();
    console.log('ZroBank signin response:', data);
    
    // Check for different possible field names for access token
    if (data.access_token) {
      this.accessToken = data.access_token;
      return data.access_token;
    }

    console.error('Available fields in response:', Object.keys(data));
    throw new Error('Access token not found in ZroBank signin response. Available fields: ' + Object.keys(data).join(', '));
  }

  /**
   * Gets transactional token from ZroBank
   * @param {string} zroBankBaseUrl - ZroBank base URL
   * @param {string} accessToken - Access token from signin
   * @returns {Promise<string>} - The transactional token
   */
  async getTransactionalToken(zroBankBaseUrl, accessToken) {
    const nonce = this.generateUUID();
    const generateTokenUrl = `${zroBankBaseUrl}/v1/payin/three-ds/authentications/generate-token`;
    
    console.log('Getting transactional token from ZroBank...', generateTokenUrl);
    
    const response = await fetch(generateTokenUrl, {
      method: 'POST',
      headers: {
        'accept': 'application/json',
        'nonce': nonce,
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${accessToken}`
      }
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    const { data } = await response.json();
    console.log('ZroBank transactional token response:', data);
    
    // Check for different possible field names for transactional token
    if (data.token) {
      return data.token;
    }

    console.error('Available fields in transactional token response:', Object.keys(data));
    throw new Error('Transactional token not found in ZroBank response. Available fields: ' + Object.keys(data).join(', '));
  }
}

// Global service instance
const zroBankTokenService = new ZroBankTokenService();

// Function to fetch transactionalToken from ZroBank
async function getTransactionalTokenZroBank(zroBankBaseUrl, zroBankApiId, zroBankApiKey) {
  return await zroBankTokenService.getTransactionalTokenZroBank(zroBankBaseUrl, zroBankApiId, zroBankApiKey);
}
