import express from "express";
import path from "path";
import { fileURLToPath } from "url";
import dotenv from "dotenv";

// Load environment variables from .env file
dotenv.config();

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();
const PORT = 3030;

// In-memory storage for 3DS return data
let returnData = {
  TransactionId: null,
  Response: null,
  MD: null,
  timestamp: null
};

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true })); // To process form data
app.use(express.static(path.join(__dirname, "public")));

// CORS middleware to allow frontend requests
app.use((req, res, next) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
  res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization');
  
  if (req.method === 'OPTIONS') {
    res.sendStatus(200);
  } else {
    next();
  }
});

// 3DS Base URL configuration
const threeDSBaseUrl = process.env.THREEDS_BASE_URL;
console.log('3DS Base URL loaded from .env:', threeDSBaseUrl);

// POST endpoint to receive 3DS return
app.post('/api/3ds-return', (req, res) => {
  console.log('Received 3DS return:', req.body);
  
  // Stores received data
  returnData = {
    TransactionId: req.body.TransactionId || req.body.transactionId || null,
    Response: req.body.Response || null,
    MD: req.body.MD || null,
    timestamp: new Date().toISOString()
  };
  
  console.log('Stored data:', returnData);
  
  res.send(returnData);
});

// GET endpoint to retrieve stored data
app.get('/api/3ds-return', (req, res) => {  
  res.json({
    success: true,
    data: returnData
  });
});

// DELETE endpoint to clear stored data
app.delete('/api/3ds-return', (req, res) => {
  console.log('Clearing 3DS return data');
  returnData = {
    TransactionId: null,
    Response: null,
    MD: null,
    timestamp: null
  };
  res.json({
    success: true,
    message: '3DS return data cleared successfully'
  });
});

// API Proxy - Execute on server (To avoid CORS error, the frontend will call the local JS server to redirect to 3DS)
app.post('/api/proxy/v2/threeds-authentication-setup', async (req, res) => {
  try {
    const threeDSAuthenticationSetupUrl = `${threeDSBaseUrl}/v2/threeds-authentication-setup`;
    console.log('Proxying request to:', threeDSAuthenticationSetupUrl);
    
    const response = await fetch(threeDSAuthenticationSetupUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json; charset=utf-8',
        'Authorization': req.headers.authorization,
      },
      body: JSON.stringify(req.body)
    });

    const result = await response.text();
    const responseData = JSON.parse(result);
    
    res.json(responseData);
  } catch (error) {
    console.error('Proxy error:', error);
    res.status(500).json({ error: 'Proxy error' });
  }
});

app.post('/api/proxy/v2/threeds-authentication', async (req, res) => {
  try {
    const threeDSAuthenticationUrl = `${threeDSBaseUrl}/v2/threeds-authentication`;
    console.log('Proxying request to:', threeDSAuthenticationUrl);
    console.log('Request body:', req.body);

    console.log('Request header:', req.headers.authorization);
    const response = await fetch(threeDSAuthenticationUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json; charset=utf-8',
        'Authorization': req.headers.authorization,
      },
      body: JSON.stringify(req.body)
    });

    const result = await response.text();
    console.log('Response:', result);

    const responseData = JSON.parse(result);
    
    res.json(responseData);
  } catch (error) {
    console.error('Proxy error:', error);
    res.status(500).json({ error: 'Proxy error' });
  }
});

app.post('/api/proxy/v2/threeds-challenge-result', async (req, res) => {
  try {
    const threeDSChallengeResultUrl = `${threeDSBaseUrl}/v2/threeds-challenge-result`;
    console.log('Proxying request to:', threeDSChallengeResultUrl);
    
    const response = await fetch(threeDSChallengeResultUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json; charset=utf-8',
        'Authorization': req.headers.authorization,
      },
      body: JSON.stringify(req.body)
    });

    const result = await response.text();
    const responseData = JSON.parse(result);
    
    res.json(responseData);
  } catch (error) {
    console.error('Proxy error:', error);
    res.status(500).json({ error: 'Proxy error' });
  }
});

app.listen(PORT, () => {
  console.log(`Server running at http://localhost:${PORT}`);
});
